// lmsshield.js
//
// Dependencies: calls SCORM API functions - so they will need to be made available.
//               findobject.js
//
// This File is designed to shield the developer
// from the SCORM 1.2 API specifics, and act as
// a more abstract interface between the LMS and
// the LMS.

function LMSShield(winRef, LMSCritical)
{
	this._winRef = window;
	this.apiWrapper = new APIWrapper(winRef, LMSCritical, true, true);
}
//////////////////////
// status constants //
//////////////////////

LMSShield.prototype.STATUS_NOT_ATTEMPTED = 0;
LMSShield.prototype.STATUS_INCOMPLETE = 1;
LMSShield.prototype.STATUS_COMPLETED = 2;
LMSShield.prototype.STATUS_PASSED = 3;
LMSShield.prototype.STATUS_FAILED = 4;
LMSShield.prototype.STATUS_UNKNOWN = 5;
	
////////////////////////////
// interactions constants //
////////////////////////////
LMSShield.prototype.INT_TRUE_FALSE = 0;
LMSShield.prototype.INT_CHOICE = 1;
LMSShield.prototype.INT_FILL_IN = 2;
LMSShield.prototype.INT_NUMERIC = 3;
LMSShield.prototype.INT_LIKERT = 4;
LMSShield.prototype.INT_MATCHING = 5;
LMSShield.prototype.INT_PERFORMANCE = 6;
LMSShield.prototype.INT_SEQUENCING = 7;
LMSShield.prototype.INT_CORRECT = 8;
LMSShield.prototype.INT_WRONG = 9;
	
///////////////////////////
// Lesson mode constants //
///////////////////////////
LMSShield.prototype.MODE_NORMAL = 0;
LMSShield.prototype.MODE_BROWSE = 1;
LMSShield.prototype.MODE_REVIEW = 2;
	
//////////////////////////
// Entry code constants //
//////////////////////////
LMSShield.prototype.ENTRY_INIT = 0;
LMSShield.prototype.ENTRY_RESUME = 1;
LMSShield.prototype.ENTRY_NONE = 2;

/////////////////////////
// Exit code constants //
/////////////////////////
LMSShield.prototype.EXIT_NONE = 0;
LMSShield.prototype.EXIT_SUSPEND = 1;
LMSShield.prototype.EXIT_LOGOUT = 2;
LMSShield.prototype.EXIT_TIMEOUT = 3;

//////////////////////
// Member Functions //
//////////////////////
LMSShield.prototype.getLessonSuccessStatus = _getLessonSuccessStatus;
LMSShield.prototype.setLessonSuccessStatus = _setLessonSuccessStatus;
LMSShield.prototype.getLessonCompletionStatus = _getLessonCompletionStatus;
LMSShield.prototype.setLessonCompletionStatus = _setLessonCompletionStatus;
LMSShield.prototype.initLMSComm = _initLMSComm;
LMSShield.prototype.terminateLMSComm = _terminateLMSComm;
LMSShield.prototype.setLessonScore = _setLessonScore;
LMSShield.prototype.getLessonScore = _getLessonScore;
LMSShield.prototype.objectivesSupported = _objectivesSupported;
LMSShield.prototype.getNextFreeObjectiveIndex = _getNextFreeObjectiveIndex;
LMSShield.prototype.initObjective = _initObjective;
LMSShield.prototype.getObjectiveSuccessStatus = _getObjectiveSuccessStatus;
LMSShield.prototype.setObjectiveSuccessStatus = _setObjectiveSuccessStatus;
LMSShield.prototype.getObjectiveCompletionStatus = _getObjectiveCompletionStatus;
LMSShield.prototype.setObjectiveCompletionStatus = _setObjectiveCompletionStatus;
LMSShield.prototype.setObjectiveScore = _setObjectiveScore;
LMSShield.prototype.getObjectiveScore = _getObjectiveScore;
LMSShield.prototype.interactionsSupported = _interactionsSupported;
LMSShield.prototype.getNextFreeInteractionIndex = _getNextFreeInteractionIndex;
LMSShield.prototype.initInteraction = _initInteraction;
LMSShield.prototype.scoreInteraction = _scoreInteraction;
LMSShield.prototype.getSuspendData = _getSuspendData;
LMSShield.prototype.setSuspendData = _setSuspendData;
LMSShield.prototype.getLessonLocation = _getLessonLocation;
LMSShield.prototype.setLessonLocation = _setLessonLocation;
LMSShield.prototype.getLessonMode = _getLessonMode;
LMSShield.prototype.getEntryCode = _getEntryCode;
LMSShield.prototype.setExitCode = _setExitCode;
LMSShield.prototype.commitData = _commitData;

/////////////////////
// LMSInitialize() //
/////////////////////
function _initLMSComm()
{
	var retVal = this.apiWrapper.LMSInitialize("");
	switch (retVal)
	{
		case "true":retVal=true;break;
		case "false":retVal=false;break;
		default:retVal=false;break;
	}
	return retVal;
}

function _terminateLMSComm()
{
	var retVal = this.apiWrapper.LMSFinish("");
	switch (retVal)
	{
		case "true":retVal=true;break;
		case "false":retVal=false;break;
		default:retVal=false;break;
	}
	return retVal;
}

function _commitData()
{
	var retVal = this.apiWrapper.LMSCommit("");
	switch (retVal)
	{
		case "true":retVal=true;break;
		case "false":retVal=false;break;
		default:retVal=false;break;
	}
	return retVal;
}

////////////////////////////
// cmi.core.lesson_status //
////////////////////////////
function _setLessonSuccessStatus(successStatus)
{
	var passed = this.STATUS_PASSED;
	var failed = this.STATUS_FAILED;

	switch (successStatus)
	{
		case 3:this.apiWrapper.LMSSetValue("cmi.core.lesson_status","passed");break;
		case 4:this.apiWrapper.LMSSetValue("cmi.core.lesson_status","failed");break;
	}
}

function _getLessonSuccessStatus()
{
	var successStatus = this.apiWrapper.LMSGetValue("cmi.core.lesson_status");
	switch (successStatus.toLowerCase())
	{
		case "passed":return this.STATUS_PASSED;break;
		case "failed":return this.STATUS_FAILED;break;
		default:return this.STATUS_UNKNOWN;break;
	}
}

function _setLessonCompletionStatus(completionStatus)
{
	if (this.getLessonSuccessStatus() == this.STATUS_UNKNOWN)
	{
		switch (completionStatus)
		{
			case 0:this.apiWrapper.LMSSetValue("cmi.core.lesson_status","not attempted");break;
			case 1:this.apiWrapper.LMSSetValue("cmi.core.lesson_status","incomplete");break;
			case 2:this.apiWrapper.LMSSetValue("cmi.core.lesson_status","completed");break;
		}
	}
}

function _getLessonCompletionStatus()
{
	var completionStatus = this.apiWrapper.LMSGetValue("cmi.core.lesson_status");
	switch (completionStatus.toLowerCase())
	{
		case "passed":return this.STATUS_COMPLETED;break;
		case "failed":return this.STATUS_COMPLETED;break;
		case "not attempted":return this.STATUS_NOT_ATTEMPTED;break;
		case "incomplete":return this.STATUS_INCOMPLETE;break;
		case "completed":return this.STATUS_COMPLETED;break;
		default:return this.STATUS_UNKNOWN;break;
	}
}

////////////////////////
// cmi.core.score.raw //
////////////////////////
function _setLessonScore(score)
{
	if (score < -100) score = -100;
	if (score > 100) score = 100;
	retVal = this.apiWrapper.LMSSetValue("cmi.core.score.raw",(score + ""));
	return (retVal == "true");
}

function _getLessonScore()
{
	var score = this.apiWrapper.LMSGetValue("cmi.core.score.raw");
	if (score == "") score = "0";
	return parseInt(score);
}

////////////////////
// cmi.objectives //
////////////////////
function _objectivesSupported()
{
	var childString = this.apiWrapper.LMSGetValue("cmi.objectives._children");
	var regExps = [new RegExp("(\\s*|,)id(\\s*|,)","i"), new RegExp("(\\s*|,)score(\\s*|,)","i"), new RegExp("(\\s*|,)status(\\s*|,)","i")];
	for (i=0;i<regExps.length;i++)
	{
		if (childString.search(regExps[i]) < 0)
		{
			return false;
		}
	}
	return true;
}

function _getNextFreeObjectiveIndex()
{
	if (this.objectivesSupported())
	{
		var retVal = parseInt(this.apiWrapper.LMSGetValue("cmi.objectives._count"));
		if (retVal == "")
		{
			retVal = 0;
		}
		return retVal;
	}
	else
	{
		return 0;
	}
}

function _initObjective(objIndex, objId)
{
	if (this.objectivesSupported())
	{
		this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".id", objId);
		this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status", "not attempted");
	}
}

function _setObjectiveSuccessStatus(objIndex, successStatus)
{
	if (this.objectivesSupported())
	{
		switch (successStatus)
		{
			case 3:this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status","passed");break;
			case 4:this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status","failed");break;
		}
	}
}

function _getObjectiveSuccessStatus(objIndex)
{
	if (this.objectivesSupported())
	{
		var successStatus = this.apiWrapper.LMSGetValue("cmi.objectives." + objIndex + ".status");
		switch (successStatus.toLowerCase())
		{
			case "passed":return this.STATUS_PASSED;break;
			case "failed":return this.STATUS_FAILED;break;
			default:return this.STATUS_UNKNOWN;break;
		}
	}
}

function _setObjectiveCompletionStatus(objIndex, completionStatus)
{
	if (this.objectivesSupported())
	{
		if (this.getObjectiveSuccessStatus() == this.STATUS_UNKNOWN)
		{
			switch (completionStatus)
			{
				case 0:this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status","not attempted");break;
				case 1:this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status","incomplete");break;
				case 2:this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".status","completed");break;
			}
		}
	}
}

function _getObjectiveCompletionStatus(objIndex)
{
	if (this.objectivesSupported())
	{
		var completionStatus = this.apiWrapper.LMSGetValue("cmi.objectives." + objIndex + ".status");
		switch (completionStatus.toLowerCase())
		{
			case "passed":return this.STATUS_COMPLETED;break;
			case "failed":return this.STATUS_COMPLETED;break;
			case "not attempted":return this.STATUS_NOT_ATTEMPTED;break;
			case "incomplete":return this.STATUS_INCOMPLETE;break;
			case "completed":return this.STATUS_COMPLETED;break;
			default:return this.STATUS_UNKNOWN;break;
		}
	}
}

function _setObjectiveScore(objIndex, score)
{
	if (score < -100) score = -100;
	if (score > 100) score = 100;
	if (this.objectivesSupported())
	{
		this.apiWrapper.LMSSetValue("cmi.objectives." + objIndex + ".score.raw", (score + ""));
	}
}

function _getObjectiveScore(objIndex)
{
	if (this.objectivesSupported())
	{
		return this.apiWrapper.LMSGetValue("cmi.objectives." + objIndex + ".score.raw");
	}
}

//////////////////////
// cmi.interactions //
//////////////////////
function _interactionsSupported()
{
	var childString = this.apiWrapper.LMSGetValue("cmi.interactions._children");
	var regExps = [new RegExp("(\\s*|,)id(\\s*|,)","i"), new RegExp("(\\s*|,)type(\\s*|,)","i"), new RegExp("(\\s*|,)correct_responses(\\s*|,)","i"), new RegExp("(\\s*|,)student_response(\\s*|,)","i"), new RegExp("(\\s*|,)result(\\s*|,)","i")];
	for (i=0;i<regExps.length;i++)
	{
		if (childString.search(regExps[i]) < 0)
		{
			return false;
		}
	}	
	return true;
}

function _getNextFreeInteractionIndex()
{
	if (this.interactionsSupported())
	{
		var retVal = parseInt(this.apiWrapper.LMSGetValue("cmi.interactions._count"));
		if (retVal == "")
		{
			retVal = 0;
		}
		return retVal;
	}
	else
	{
		return 0;
	}
}

function _initInteraction(intIndex, intId, intType, correctResponse)
{
	if (this.interactionsSupported())
	{
		var strType;
		var numCorrectResponses = arguments.length - 3;
		
		if (arguments.length < 2)
		{
			alert("You have supplied too few parameters in the call to initInteraction(intIndex, intId, intType, *correctResponses). At least 2 parameters are required. (you supplied " + arguments.length + ")");
			return;
		}
		// Set the interactions id.
		this.apiWrapper.LMSSetValue("cmi.interactions." + intIndex + ".id", intId);
		// Determine and set the interaction type.
		switch (intType)
		{
			case 0:strType="true-false";break;
			case 1:strType="choice";break;
			case 2:strType="fill-in";break;
			case 3:strType="numeric";break;
			case 4:strType="likert";break;
			case 5:strType="matching";break;
			case 6:strType="performance";break;
			case 7:strType="sequencing";break;
			default:strType="";break;
		}
		if (strType != "")
		{
			this.apiWrapper.LMSSetValue("cmi.interactions." + intIndex + ".type", strType);
		}
		
		// Set any correct responses that may have been specified.
		for (var i=0;i<numCorrectResponses;i++)
		{
			this.apiWrapper.LMSSetValue("cmi.interactions." + intIndex + ".correct_responses." + i + ".pattern", arguments[3+i])
		}
	}
}

function _scoreInteraction(intIndex, intResult, studentResponse)
{
	if (this.interactionsSupported())
	{
		if (arguments.length < 2)
		{
			alert("You have supplied too few parameters in the call to scoreInteraction(intIndex, intResult, studentResponse). At least 2 parameters are required. (you supplied " + arguments.length + ")");
			return;
		}
		
		// Determine and set result.
		switch (intResult)
		{
			case 8:intResult="correct";break;
			case 9:intResult="wrong";break;
			default: alert("You have supplied an invalid intResult parameter in the call to scoreInteraction(intIndex, intResult, studentResponse). The value you supplied is: " + intResult + ".");return;
		}
		this.apiWrapper.LMSSetValue("cmi.interactions." + intIndex + ".result", intResult);
		
		// Set student response.
		if (arguments.length > 2)
		{
			this.apiWrapper.LMSSetValue("cmi.interactions." + intIndex + ".student_response", studentResponse);
		}
	}
}

//////////////////////
// cmi.suspend_data //
//////////////////////
function _getSuspendData()
{
	return this.apiWrapper.LMSGetValue("cmi.suspend_data");
}

function _setSuspendData(value)
{
	return this.apiWrapper.LMSSetValue("cmi.suspend_data", value);
}

//////////////////////////////
// cmi.core.lesson_location //
//////////////////////////////
function _getLessonLocation()
{
	return this.apiWrapper.LMSGetValue("cmi.core.lesson_location");
}

function _setLessonLocation(value)
{
	return this.apiWrapper.LMSSetValue("cmi.core.lesson_location", value);
}

//////////////////////////
// cmi.core.lesson_mode //
//////////////////////////
function _getLessonMode()
{
	var mode =  this.apiWrapper.LMSGetValue("cmi.core.lesson_mode");
	var retVal;
	switch (mode)
	{
		case "normal":retVal = this.MODE_NORMAL;break;
		case "review":retVal = this.MODE_REVIEW;break;
		case "browse":retVal = this.MODE_BROWSE;break;
		default:retVal = this.MODE_NORMAL;break;
	}
	return retVal;
}

////////////////////
// cmi.core.entry //
////////////////////
function _getEntryCode()
{
	var entryCode =  this.apiWrapper.LMSGetValue("cmi.core.entry");
	var retVal;
	switch (entryCode)
	{
		case "ab-initio":retVal = this.ENTRY_INIT;break;
		case "resume":retVal = this.ENTRY_RESUME;break;
		default:retVal = this.ENTRY_NONE;break;
	}
	return retVal;
}

///////////////////
// cmi.core.exit //
///////////////////
function _setExitCode(exitCode)
{
	var retVal = false;
	switch (exitCode)
	{
		case 0:retVal = this.apiWrapper.LMSSetValue("cmi.core.exit","");break;
		case 1:retVal = this.apiWrapper.LMSSetValue("cmi.core.exit","suspend");break;
		case 2:retVal = this.apiWrapper.LMSSetValue("cmi.core.exit","logout");break;
		case 3:retVal = this.apiWrapper.LMSSetValue("cmi.core.exit","time-out");break;
	}
	
	return retVal;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

function TimeSpan()
{
	this.startTime = new Date();
}
TimeSpan.prototype.MILLISECONDS = 0;
TimeSpan.prototype.CMITIMESPAN = 1;
TimeSpan.prototype.getElapsed = TimeSpanGetElapsed;
TimeSpan.prototype.getTimespan = TimeSpanGetTimeSpan;

// This member function of the TimeSpan
// object will return, in SCORM CMITimespan
// form, the elapsed time from the time the
// object was created to the time the function
// was called.
function TimeSpanGetElapsed(timeFormat)
{
	if (timeFormat == null)
	{
		timeFormat = this.MILLISECONDS;
	}
	var retVal = "";
	var milliseconds = 0;
	
	// Set milliseconds equal to the number of milliseconds
	// between creation of this object and this function call.
	milliseconds = (new Date()).getTime() - this.startTime.getTime();
	
	switch (timeFormat)
	{
		case 0:return milliseconds;
		case 1:return this.getTimespan(milliseconds);
		default:alert("Illegal format supplied in call to TimeSpan.getElapsed().");return 0;
	}
}

function TimeSpanGetTimeSpan(milliseconds)
{
	var retVal = "";
	var hours = 0;
	var minutes = 0;
	var seconds = 0;
	var hundreths = 0;
	
	hours = Math.floor(milliseconds/3600000);
	milliseconds -= hours*3600000;

	minutes = Math.floor(milliseconds/60000);
	milliseconds -= minutes*60000;

	seconds = Math.floor(milliseconds/1000);
	milliseconds -= seconds*1000;

	hundreths = Math.floor(milliseconds/10);
	milliseconds -= hundreths*10;
	
	if (hours > 9999)
	{
		hours = 0;
	}
	
	if (hours < 10)
	{
		hours = "0" + hours;
	}
	else
	{
		hours += "";
	}
	
	if (minutes < 10)
	{
		minutes = "0" + minutes;
	}
	else
	{
		minutes += "";
	}
	
	if (seconds < 10)
	{
		seconds = "0" + seconds;
	}
	else
	{
		seconds += "";
	}
	
	if (hundreths < 10)
	{
		hundreths = "0" + hundreths;
	}
	else
	{
		hundreths +="";
	}
	
	return hours + ":" + minutes + ":" + seconds + "." + hundreths;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////
// SCORM Version 1.2 API Wrapper //
///////////////////////////////////

///////////////////////////////////////////////////////////
// Purpose:                                              //
// Find the LMS API Adapter (will be null if not found), //
// and define API wrapper functions.  If the API         //
// Adapter is not found, then act appropriately by       //
// either closing the lesson window, or allowing the     //
// lesson to continue in its absence.                    //
///////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////
// Wrapper Functions:                                            //
// String LMSInitialize()                                        //
// String LMSFinish()                                            //
// String LMSSetValue(String data_model_element, String value)   //
// String LMSGetValue(String data_model_element)                 //
// String LMSCommit()                                            //
// String LMSGetLastError()                                      //
// String LMSGetErrorString(String errorNumber)                  //
// String LMSGetDiagnostic(Optional String errorNumber)          //
///////////////////////////////////////////////////////////////////

function APIWrapper(winRef, scormCritical, debugAlerts, diagnosticAlerts)
{
	
	// SCORM_DEBUG causes error messages to be displayed if any
	// API Adapter errors are encountered after any API call made.
	this.SCORM_DEBUG = debugAlerts;
	// SCORM_CRITICAL causes the lesson to be closed if any
	// API Adapter errors are encountered after any API call made.
	this.SCORM_CRITICAL = scormCritical;
	// DISPLAY_DIAGNOSTIC is a switch that allows information from
	// LMSGetDiagnostic() to be displayed in error messages.  If 
	// is set to false, LMSGetDiagnostic() will not be called
	// when displaying error messages.
	this.DISPLAY_DIAGNOSTIC = diagnosticAlerts;
	
	this.API = null;
	
	// sessionTime is TimeSpan object 
	this.sessionTime = null;
	// inLMSFinish is a flag that prevents the LMSFinish wrapper
	// from being called twice.
	this.inLMSFinish = false;
	
	this.ABORT = false;
	this.enableCommit = true;
	
	// obtain the API Adapter reference
	this.API = findObject(winRef, "API");
	
	// verify the existance of the API Adapter, and take any action needed.
	if (this.API == null && this.SCORM_CRITICAL)
	{
		alert("FATAL ERROR: Cannot locate SCORM API object. \nThis lesson will now be closed.");
		winRef.parent.close();
	}
}
APIWrapper.prototype.LMSInitialize = _LMSInitialize;
APIWrapper.prototype.LMSFinish = _LMSFinish;
APIWrapper.prototype.LMSGetValue = _LMSGetValue;
APIWrapper.prototype.LMSSetValue = _LMSSetValue;
APIWrapper.prototype.LMSCommit = _LMSCommit;
APIWrapper.prototype.LMSGetLastError = _LMSGetLastError;
APIWrapper.prototype.LMSGetErrorString = _LMSGetErrorString;
APIWrapper.prototype.LMSGetDiagnostic = _LMSGetDiagnostic;

///////////////////////////
// API Wrapper Functions //
///////////////////////////

function _LMSInitialize()
{
	if (this.ABORT) return;
	this.sessionTime = new TimeSpan();
	if (this.API != null)
	{
		var retVal = this.API.LMSInitialize("").toString();
		var lastError = this.LMSGetLastError();
		if (lastError != "0" && this.SCORM_CRITICAL)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + Lthis.MSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic());
			}
			else
			{ 
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError));
			}
			this.ABORT = true;
			return "false";
		}
		else if (lastError != "0" && this.SCORM_DEBUG)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic());
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError));
			}
			return "false";
		}
		return retVal;
	}
	else
	{
		return "true";
	}
}

function _LMSFinish()
{
	if (this.ABORT) return;
	
	if (!this.inLMSFinish)
	{
		this.inLMSFinish = true;
		this.LMSSetValue("cmi.core.session_time",this.sessionTime.getElapsed(this.sessionTime.CMITIMESPAN));
		this.LMSCommit();
		if (this.API != null)
		{
			return this.API.LMSFinish("").toString();
		}
		else
		{
			return "true";
		}
	}
}

function _LMSGetValue(element)
{
	if (this.ABORT) return;
	
	if (this.API != null)
	{
		var retVal = this.API.LMSGetValue(element).toString();
		var lastError = this.LMSGetLastError();
		if (lastError != "0" && this.SCORM_CRITICAL)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic() + "\nThis lesson will now be closed.");
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\nThis lesson will now be closed.");
			}
			this.LMSFinish();
			this.ABORT = true;
		}
		else if (lastError != "0" && this.SCORM_DEBUG)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic());
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError));
			}
		}
		return retVal;
	}
	else
	{
		return "";
	}
}

function _LMSSetValue(element, value) 
{
	if (this.ABORT) return;
	
	var retVal = "true";

	if (this.API != null)
	{
		retVal = this.API.LMSSetValue(element, value).toString();
		var lastError = this.LMSGetLastError();
		if (lastError != "0" && this.SCORM_CRITICAL)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic() + "\nThis lesson will now be closed.");
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\nThis lesson will now be closed.");
			}
			this.LMSFinish();
			this.ABORT = true;
		}
		else if (lastError != "0" && this.SCORM_DEBUG)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic());
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError));
			}
		}
		
		return retVal;
	}
	return retVal;
}

function _LMSCommit()
{
	if (this.ABORT) return;
	
	if (this.API != null)
	{
		var retVal = this.API.LMSCommit("").toString();
		var lastError = this.LMSGetLastError();
		if (lastError != "0" && this.SCORM_CRITICAL)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic() + "\nThis lesson will now be closed.");
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\nThis lesson will now be closed.");
			}
			
			this.LMSFinish();
			this.ABORT = true;
		}
		else if (lastError != "0" && this.SCORM_DEBUG)
		{
			if (this.DISPLAY_DIAGNOSTIC)
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError) + "\n" + this.LMSGetDiagnostic());
			}
			else
			{
				alert("FATAL SCORM ERROR: " + this.LMSGetErrorString(lastError));
			}
		}
		return retVal;
	}
	else
	{
		return "true";
	}
}

function _LMSGetLastError()
{
	if (this.ABORT) return;
	
	if (this.API != null)
	{
		return this.API.LMSGetLastError().toString();
	}
	else
	{
		return "0";
	}
}

function _LMSGetErrorString(errornumber)
{
	if (this.ABORT) return;
	
	if (this.API != null)
	{
		return this.API.LMSGetErrorString(errornumber + "").toString();
	}
	else
	{
		return "No errors detected";
	}
}

function _LMSGetDiagnostic(parameter)
{
	if (this.ABORT) return;
	
	if (this.API != null)
	{
		if (parameter == null)
		{
			return this.API.LMSGetDiagnostic("").toString();
		}
		else
		{
			return this.API.LMSGetDiagnostic(parameter + "").toString();
		}
	}
	else
	{
		return "No errors detected";
	}
}
